---
sidebar_position: 3
title: User Account and Points Retrieval
slug: FPVY464Iqc2ZCl1pANyS0
createdAt: Thu Jul 18 2024 05:56:40 GMT+0000 (Coordinated Universal Time)
updatedAt: Fri Jul 19 2024 03:50:33 GMT+0000 (Coordinated Universal Time)
---

# Get User Account Function

This documentation provides a detailed explanation of the `getAccount` function, which handles user account retrieval, login authentication, and interaction with the Ethereum provider. It includes asynchronous operations to ensure smooth and non-blocking user experience.

## Function: `getAccount`

### Description

The `getAccount` function checks if a user is logged in. If not, it initiates the login process. Once the user is logged in, it retrieves the user's Ethereum account. The function also handles loading indicators and emits events upon successful account retrieval.

### Code

```javascript
async function getAccount() {
	try {
		// Create a new User instance
		let user = new User();
		
		// Check if the user is logged in
		const isLogin_status = await user.islogin();
		console.log(isLogin_status, 'isLogin_status');
		
		if (!isLogin_status) {
			// Show loading indicator if the user is not logged in
			showLoadingToast({
				duration: 0,
				forbidClick: true,
				message: 'Loading...',
			});
			
			// Initiate login process
			user.login((acc, userId, error) => {
				// Close loading indicator
				closeToast();
				console.log("getWeb3 account", acc);
				console.log("getWeb3 userId", userId);
				console.log("getWeb3 error", error);
				
				// Set the account value and emit event upon successful login
				account.value = acc;
				bus.emit('get_balance', "login");
			});
		} else {
			// If user is already logged in, interact with Ethereum provider to get account
			let account = await user.getAccount();
			console.log("getWeb3 account", account);
			let userId = await user.getUserId();
			console.log("getWeb3 userId", userId);
			account.value = account;
			
			// Emit event after retrieving the account
			bus.emit('get_balance', "login");
		}
	} catch (error) {
		// Close loading indicator and handle errors
		closeToast();
		console.log(error, "getAccount error");
		
		if (error && typeof error == 'string') {
			showToast(error);
		} else {
			showToast(error.message);
		}
	} finally {
		// Final clean-up actions if needed
	}
}
```

### Detailed Explanation

1. **User Object Initialization:**
   - Creates a new instance of the `User` object.
2. **Check Login Status:**
   - Asynchronously checks if the user is logged in.
   - Logs the login status.
3. **Handle Not Logged In Status:**
   - Shows a loading toast if the user is not logged in.
4. **User Login Process:**
   - Initiates the login process.
   - Closes the loading toast upon completion.
   - Logs the account, userId, and any error.
   - Sets the account value and emits an event upon successful login.
5. **Handle Logged In Status:**
   - Interacts with the Ethereum provider to get the account if the user is already logged in.
   - Sets the account value and emits an event after retrieving the account.
6. **Error Handling:**
   - Closes the loading toast in case of an error.
   - Logs the error.
   - Shows an appropriate error message to the user.

### Usage in Component Lifecycle

The `getAccount` function can be called within the component's lifecycle hooks to ensure that the user's account is retrieved and logged in status is checked when necessary.

### Example Usage

```javascript
onMounted(() => {
    getAccount();
});
```

This setup ensures that the user account status is checked and handled as soon as the component is ready, providing a seamless experience for the user.

